;
/*************************************************************
 *
 * Copyright (c) 2023 ysrock Co., Ltd.	<info@ysrock.co.jp>
 * Copyright (c) 2023 Yasuo Sugano	<sugano@ysrock.co.jp>
 *
 * Version	: 0.0.3
 * Update	  : 2023.12.05
 *
 ************************************************************/
'use strict';

  /**
 * 分割アップローダー
 */
window['splitUploader'] = (function(){
  return{
    ready: function() {

    },

    /**
     * ドラッグ＆ドロップエリアを設定
     *
     *  window.splitUploader.setDropArea( $('#drop'), 'https://demo.ysrock.com/ys_uploader/upload.php', optionObject );
     *  window.splitUploader.setDropArea( document.getElementsByClassName('CLASS'), 'https://demo.ysrock.com/ys_uploader/upload.php', optionObject );
     *
     *  @param ドラッグ＆ドロップを有効にするエレメント
     *  @param アップロード先PHPのURL
     *  @param optionObject
     *    mimeAllow: [ 'image/jpeg', 'image/png'... ]
     *      MIME制限
     *    splitSize: 1048576
     *      分割するサイズ
     *    cbMimeFail: function(HTMLInputElement.files) {}
     *      MIME制限にかかった場合のコールバック
     *    cbReadPrepare: function( result )
     *      ファイルを読み込む直前のコールバック
     *    cbReadDone: function( result )
     *      ファイルを読み込んだ直後のコールバック
     *    cbUploadDone: function( result )
     *      アップロード完了
     *    cbUploadFail: function( result, error )
     *      アップロードエラー
     */
    setDropArea: function(elements, phpPath, optionObject) {
      let reElements = [];
      // 引数が配列
      if (Array.isArray(elements)) reElements = elements;
      // クラス等で複数存在
      else if (elements[0]) {
        for (let i=0, len=elements.length; i<len; i++) reElements.push(elements[i]);
      } else {
        reElements = [ elements ];
      };

      if (typeof optionObject == "undefined") optionObject = {};
      

      for (let i=0, len=reElements.length; i<len; i++) {
        // イベント：ドロップ
        this.setDrop(reElements[i], phpPath, optionObject);
        // イベント：ドラッグオーバー
        this.setDragOver(reElements[i]);
        // イベント：ドラッグ終了
        this.setDragLeave(reElements[i]);
      };// END for

    },
    // イベント：ドロップ
    setDrop: function(elem, phpPath, optionObject) {
      elem.addEventListener('drop', (e) => {
        e.stopPropagation();
        e.preventDefault();
        window.splitUploader.readData(e.dataTransfer, elem, phpPath, optionObject);
      });
    },
    // イベント：ドラッグオーバー
    setDragOver: function(elem) {
      elem.addEventListener('dragover', (e) => {
        e.stopPropagation();
        e.preventDefault();
      });
    },
    // イベント：ドラッグ終了
    setDragLeave: function(elem) {
      elem.addEventListener('dragleave', (e) => {
        e.stopPropagation();
        e.preventDefault();
      });
    },


    /**
     * INPUT[type=file] だけで分割アップロード機能を持たせる
     *
     *  window.splitUploader.setInputFile( $('#input'), 'https://demo.ysrock.com/ys_uploader/upload.php', optionObject );
     *  window.splitUploader.setInputFile( document.getElementsByClassName('CLASS'), 'https://demo.ysrock.com/ys_uploader/upload.php', optionObject );
     *
     *  @param ドラッグ＆ドロップを有効にするエレメント
     *  @param アップロード先PHPのURL
     *  @param optionObject
     *    mimeAllow: [ 'image/jpeg', 'image/png'... ]
     *      MIME制限
     *    splitSize: 1048576
     *      分割するサイズ
     *    cbMimeFail: function(HTMLInputElement.files) {}
     *      MIME制限にかかった場合のコールバック
     *    cbReadPrepare: function( chunks )
     *      ファイルを読み込む直前のコールバック
     *    cbReadDone: function( chunks )
     *      ファイルを読み込んだ直後のコールバック
     *    cbUploadDone: function( result )
     *      アップロード完了
     *    cbUploadFail: function( chunks, error )
     *      アップロードエラー
     */
    setInputFile: function(elements, phpPath, optionObject) {
      let reElements = [];
      // 引数が配列
      if (Array.isArray(elements)) reElements = elements;
      // クラス等で複数存在
      else if (elements[0]) {
        for (let i=0, len=elements.length; i<len; i++) reElements.push(elements[i]);
      } else {
        reElements = [ elements ];
      };

      if (typeof optionObject == "undefined") optionObject = {};

      for (let i=0, len=reElements.length; i<len; i++) {
        reElements[i].addEventListener('change', (e) => {
          window.splitUploader.readData(reElements[i], reElements[i], phpPath, optionObject);
        });
      };// END for
    },


    /**
     * ファイルを読み込む
     */
    readData: function(fileElement, triggerElement, phpPath, optionObject) {
      // 分割するサイズ
      if('splitSize' in optionObject === false && typeof optionObject.splitSize != "number") optionObject.splitSize = 1048576;
      // ファイルをひとつずつ処理する
      for (let i=0, len=fileElement.files.length; i<len; i++) {
        let file = fileElement.files[i];
        // MIMEチェック
        if (!window.splitUploader.checkMIME(file, optionObject)) continue;
        // UUIDを作成
        let uuid = window.splitUploader.getUUID();
        // 送信する分割回数
        let denominator = Math.ceil(file.size / optionObject.splitSize) - 1;
        // チャンク
        let chunks = {
          'uuid' : uuid,
          'file' : file,
          'triggerElement' : triggerElement,
          'splitSize' : optionObject.splitSize,
          'denominator' : denominator
        };

        // ファイルを読み込む直前のコールバック
        if (typeof optionObject.cbReadPrepare == "function") optionObject.cbReadPrepare( chunks );

        // FileReader
        let FR = new FileReader();
        FR.addEventListener('load', function(e) {
          let chunks = e.target.chunks;
          chunks.base64 = e.target.result;
          // ファイルを読み込んだ直後のコールバック
          if (typeof optionObject.cbReadDone == "function") optionObject.cbReadDone( chunks );
          // 分割してアップロード
          chunks.counter = 0;
          window.splitUploader.splitFile(phpPath, optionObject, chunks);
        });
        FR.chunks = chunks;
        FR.readAsDataURL(file);
        // console.log([ 'split-uploader.js', 'readData', true  ]);

      };// END for
    },

    /**
     * 分割してアップロード
     */
    splitFile: function(phpPath, optionObject, chunks) {
      // 分割開始位置
      let begin = Number(chunks.counter) * Number(chunks.splitSize);
      // 分割終了位置
      let end = Number(begin) + Number(chunks.splitSite);
      // FileReader
      let FR = new FileReader();
      FR.onload = function(e) {
        let chunks = e.target.chunks;
        chunks.data = e.target.result;
        // 非同期にアップロード
        window.splitUploader.ajaxUpload(phpPath, optionObject, chunks);
      };
      FR.chunks = chunks;
      FR.readAsArrayBuffer(chunks['file'].slice(begin, end));
    },

    /**
     * 非同期にアップロード
     */
    ajaxUpload: function(phpPath, optionObject, chunks) {
      fetch(phpPath, {
        method: "POST",
        headers: {
          'chunk-uuid': chunks.uuid,
          'chunk-counter': chunks.counter,
          'chunk-denominator': chunks.denominator
        },
        body: chunks.data
      }).then(function(response) {
        if (!response.ok) throw new Error(`Request failed: '${response.status}`);
        return response.json();
      }).then(function(result) {
        if ('next' in result) window.splitUploader.nextChunk(phpPath, optionObject, chunks, result);
        else if (typeof optionObject.cbUploadDone == "function")  optionObject.cbUploadDone(result);
      }).catch(function(err) {
        console.error(['split-uploader.js', 'catch', err]);
        if (typeof optionObject.cbUploadFail == "function") optionObject.cbUploadFail(chunks, err);
      });
    },

    /**
     * 続けてアップロード
     */
    nextChunk: function(phpPath, optionObject, chunks, result) {
      if (typeof optionObject.cbNext == "function") optionObject.cbNext(chunks);
      chunks.counter = result.next;
      window.splitUploader.splitFile(phpPath, optionObject, chunks);
    },


    /**
     * MIMEチェック
     */
    checkMIME: function(fileObj, optionObject) {
      // 指定なし
      if (typeof optionObject != "object" || optionObject.mimeAllow === undefined) return true;
      // 文字列の場合は配列に変換する
      if (typeof optionObject.mimeAllow == "string") optionObject.mimeAllow = [ optionObject.mimeAllow ];
      // 一つでも一致すれば許可する
      for (let i=0, len=optionObject.mimeAllow.length; i<len; i++) {
        if(fileObj.type == optionObject.mimeAllow[i]) return true;
      };// END for
      // コールバックがある場合実行する
      if(typeof optionObject.cbMimeFail == "function") optionObject.cbMimeFail( file );
    },


    /**
     * UUIDを生成
     */
    getUUID: function() {
      return "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(/[xy]/g, function(c){
        let r = Math.random() * 16 | 0, v = c == "x" ? r : r & 3 | 8;
        return v.toString(16);
      });
    },


  }
})();
