;
/*************************************************************
 *
 * Copyright (c) 2026 ysrock Co., Ltd.	<info@ysrock.co.jp>
 * Copyright (c) 2026 Yasuo Sugano	<sugano@ysrock.co.jp>
 *
 * Version	: 1.0.0
 * Update		: 2026.02.06
 *
 ************************************************************/
'use strict';

class YsPasori {

  constructor( argOpt ) {
    // ペアリング対象機種リスト
    this.DEVICE_INFO_LIST = {
			3528 : {
				vendorId : 1356,
				productId: 3528,
				modelName: "RC-S300/S",
				deviceType: "External"
			},
      3529 : {
        vendorId: 1356,
        productId: 3529,
        modelName: "RC-S300/P",
        deviceType: "External"
      }
    }


    this.USBDevice = null ;
    this.productId = null;
    this.interfaces = null;
    this.endPoint = null;
    this.polling = false;
    this.seqNumber = 0;


    // ペアリング時のフィルタ
    this.DEVICE_FILTERS = [];
    for (let pid in this.DEVICE_INFO_LIST) {
      const vendorId = this.DEVICE_INFO_LIST[pid].vendorId;
      this.DEVICE_FILTERS.push({ vendorId: vendorId , productId: pid });
    }


  }


  /**
   * USBデバイス接続
   */
  async connectUsbDevice() {
    try {
      // ペアリング設定済みデバイスの取得
      this.USBDevice = await this.getDevices();
      // プロダクトIDの取得
      this.productId = this.getProductId(this.USBDevice);
      // USB デバイスオープン
      await this.usbDeviceOpen();
    } catch (e) {
      console.log(e);
      alert(e);
      throw e;
    }
  }


  /**
   * ポーリングを開始
   */
  async startPolling() {
    if (this.productId == 3528 || this.productId == 3529) {
      return await this.startPollingRCS300();
    }

  }


  /**
   * ポーリングを終了する
   */
  stopPolling() {
    this.polling = false;
  }



  /*--------------------------------------------------
   * private
   *--------------------------------------------------*/

  /**
   * ペアリング設定済みデバイスの取得
   *
   *  @return USBDevice $dev
   */
  async getDevices() {
    let usbDevice = null;

    const devAry = await navigator.usb.getDevices();

    // ペアリング済み
    let pearInt = 0;
    if (devAry.length > 0) {
      for (let dev of devAry) {
        const tempDev = this.DEVICE_FILTERS.find((filter) => dev.vendorId == filter.vendorId && dev.productId == filter.productId);
        if (tempDev !== undefined) {
          ++pearInt;
          usbDevice = dev;
        }
      }
    }
    if (pearInt == 1) return usbDevice;

    // USB機器をペアリングフローから選択しデバイスのUSBDeviceインスタンス取得
    return await navigator.usb.requestDevice({ filters: this.DEVICE_FILTERS });
  }


  /**
   * プロダクトIDの取得
   *
   *  @param USBDevice $USBDevice
   *  @return int $productId
   */
  getProductId(USBDevice) {
    const productId = USBDevice.productId;
    if (productId in this.DEVICE_INFO_LIST === false) {
      const error = ['カードリーダーがサポート外の機種です', 'プロダクトID:' + productId];
      console.error(error);
      throw error;
    }
    return productId;
  }


  /**
   * USB デバイスオープン
   */
  async usbDeviceOpen() {
    // USBデバイスの構成を選択
    const configurationValue = this.USBDevice.configuration.configurationValue;
    // 指定インターフェイスを排他アクセスにする	
    this.interfaces = this.USBDevice.configuration.interfaces[ configurationValue ];

    await this.USBDevice.open();
    await this.USBDevice.selectConfiguration(configurationValue);
    await this.USBDevice.claimInterface(this.interfaces.interfaceNumber);

    // エンドポイント
    this.endPoint = {
      in: this.interfaces.alternate.endpoints.filter(e => e.direction == 'in')[0].endpointNumber,
      out: this.interfaces.alternate.endpoints.filter(e => e.direction == 'out')[0].endpointNumber
    };
  }


  /**
   * USB デバイスクローズ
   */
  async usbDeviceClose() {
    // await this.USBDevice.releaseInterface(this.interfaces.interfaceNumber);
    await this.USBDevice.close();
  }


  /**
   * sleep
   *
   *  @param int $ms
   */
  async sleep(msec) {
    return new Promise(resolve => setTimeout(resolve, msec));
  }

  padding_zero(num, p){
    return ("0".repeat(p*1) + num).slice(-(p*1));
  }
  dec2HexString(n) {
    return this.padding_zero((n*1).toString(16).toUpperCase(),2);
  }
  


  async receive(len) {
    let data = await this.USBDevice.transferIn(this.endPoint.in, len);
    await this.sleep(10);
    let arr = [];
    let arr_str = [];
    for (let i = data.data.byteOffset; i < data.data.byteLength; i++) {
      arr.push(data.data.getUint8(i));
      arr_str.push(this.dec2HexString(data.data.getUint8(i)));
    }
    return arr;
  }



  /*--------------------------------------------------
   * RC-S300
   *--------------------------------------------------*/

  /**
   * ポーリングを開始
   *
   *  @return ?string
   */
  async startPollingRCS300() {
    try {
      this.polling = true;
      do {
        if (!this.polling) break;
        // セッション
        const ID = await this.sessionRCS300();
        if (ID) return ID;
        // 0.1秒ウェイト
        await this.sleep(100);
      } while(true);
    } catch (e) {
      console.error(e);
      alert(e);
      throw e;
    }
    return null;
  }


  /**
   * セッション
   *
   *  @return ?string
   */
  async sessionRCS300() {
    const len = 50;

    // firmware version
    await this.send300([0xFF, 0x56, 0x00, 0x00]);
    await this.receive(len);

    // end transparent
    await this.send300([0xFF, 0x50, 0x00, 0x00, 0x02, 0x82, 0x00, 0x00]);
    await this.receive(len);

    // start transparent
    await this.send300([0xFF, 0x50, 0x00, 0x00, 0x02, 0x81, 0x00, 0x00]);
    await this.receive(len);

    // rf off
    await this.send300([0xFF, 0x50, 0x00, 0x00, 0x02, 0x83, 0x00, 0x00]);
    await this.receive(len);

    // Switch Protocol TypeF
    await this.send300([0xff, 0x50, 0x00, 0x02, 0x04, 0x8f, 0x02, 0x03, 0x00, 0x00]);
    await this.receive(len);

    // FeriCa poling
    await this.send300([0xFF, 0x50, 0x00, 0x01, 0x00, 0x00, 0x11, 0x5F, 0x46, 0x04, 0xA0, 0x86, 0x01, 0x00, 0x95, 0x82, 0x00, 0x06, 0x06, 0x00, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00]);
    const poling_res_f = await this.receive(len);
    if(poling_res_f.length == 46){
      const idm = poling_res_f.slice(26,34).map(v => this.dec2HexString(v));
      const idmStr = idm.join('');
// console.log("Card Type: Felica  カードのIDm: " + idmStr);
      return idmStr;
    }

    // Switch Protocol TypeA
    // await this.send300([0xff, 0x50, 0x00, 0x02, 0x04, 0x8f, 0x02, 0x00, 0x03, 0x00]);
    // await this.receive(len);
  
    // GET Card UID
//     await this.send300([0xff, 0xCA, 0x00, 0x00]);
//     const poling_res_a = await this.receive(len);
//     if (poling_res_a.length == 16) {
//       const id = poling_res_a.slice(10,14).map(v => this.dec2HexString(v));
//       const idStr = id.join('');
// console.log("Card Type : MIFARE  カードのID: " + idStr);
//       return;
//     }

    return false;
  }


  async send300(data) {
    let argData = new Uint8Array(data);
    const dataLen = argData.length;
    const SLOT_NUMBER = 0x00;
    let retVal = new Uint8Array(10 + dataLen);

    retVal[0] = 0x6b ;                  // ヘッダー作成
    retVal[1] = 255 & dataLen ;         // length をリトルエンディアン
    retVal[2] = dataLen >> 8 & 255 ;
    retVal[3] = dataLen >> 16 & 255 ;
    retVal[4] = dataLen >> 24 & 255 ;
    retVal[5] = SLOT_NUMBER ;           // タイムスロット番号
    retVal[6] = ++this.seqNumber ;      // 認識番号

    0 != dataLen && retVal.set( argData, 10 ); // コマンド追加
    const out = await this.USBDevice.transferOut(this.endPoint.out, retVal);
    await this.sleep(50);
  }
}