;
/*************************************************************
 *
 * Copyright (c) 2025 ysrock Co., Ltd.	<info@ysrock.co.jp>
 * Copyright (c) 2025 Yasuo Sugano	<sugano@ysrock.co.jp>
 *
 * Version	: 1.0.0
 * Update	  : 2025.10.10
 *
 ************************************************************/
'use strict';

/**
 * Pagination クラス
 *
 *  ページネーションを管理・描画するクラス。
 *  前頁・次頁ボタンおよびページ番号リンク、
 *  現在の表示範囲を自動更新。
 *
 *  @example
 *    const P = new Pagination(document.getElementById('resultList'));
 */
class Pagination {
  help() {
    let TXT = " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  Copyright (c) 2025 ysrock Co., Ltd. <info@ysrock.co.jp>\n";
    TXT += " *  Copyright (c) 2025 Yasuo Sugano <sugano@ysrock.co.jp>\n";
    TXT += " *\n";
    TXT += " *  Version : 1.0.0\n";
    TXT += " *  update  : 2025.10.10\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ページネーション\n";
    TXT += " *    ページネーションを管理・描画するクラス。\n";
    TXT += " *    前頁・次頁ボタンおよびページ番号リンク、\n";
    TXT += " *    現在の表示範囲を自動更新。\n";
    TXT += " *\n";
    TXT += " *  const P = new Pagination(controlsSelector, ?infoSelector, ?totalItems, ?itemsPerPage, ?currentPage, ?callbackClickPage);\n";
    TXT += " *    @param {HTMLElement[]} controlsSelector - ページ操作UIを表示する要素\n";
    TXT += " *    @param {?HTMLElement[]} infoSelector - 件数範囲を表示する要素\n";
    TXT += " *    @param {?number} totalItems - 総件数\n";
    TXT += " *    @param {?number} itemsPerPage - １ページに表示する件数\n";
    TXT += " *    @param {?number} currentPage - 表示させるページ番号\n";
    TXT += " *    @param {?function(number)} callbackClickPage - ページ変更時のコールバック。ページ番号を渡す\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  指定したページ番号へ移動する\n";
    TXT += " *\n";
    TXT += " *  P.goToPage(page);\n";
    TXT += " *    @param {number} page - 遷移先のページ番号\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  データをリセットする\n";
    TXT += " *\n";
    TXT += " *  P.reset(totalItems, itemsPerPage, ?currentPage);\n";
    TXT += " *    @param {number} totalItems - 総件数\n";
    TXT += " *    @param {number} itemsPerPage - １ページに表示する件数\n";
    TXT += " *    @param {?number} currentPage - 表示させるページ番号\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ページ番号をクリックしたときのコールバック\n";
    TXT += " *\n";
    TXT += " *  P.onPageChange = function (page) {}\n";
    TXT += " *    @param {number} page - ページ番号\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ページ操作UIを表示する要素を追加\n";
    TXT += " *\n";
    TXT += " *  P.addControlsEl(El);\n";
    TXT += " *    @param {HTMLElement[]} El - ページ操作UIを表示する要素\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ページ操作UIを表示する要素を削除\n";
    TXT += " *\n";
    TXT += " *  P.removeControlsEl(idx);\n";
    TXT += " *    @param {number} idx - 削除したいインデックス番号\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ページ操作UIを表示する要素を取得\n";
    TXT += " *\n";
    TXT += " *  P.getControlsEl();\n";
    TXT += " *    @return {HTMLElement[]} ページ操作UIを表示する要素\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  件数範囲を表示する要素を追加\n";
    TXT += " *\n";
    TXT += " *  P.addInfoEl(El);\n";
    TXT += " *    @param {HTMLElement[]} El - 件数範囲を表示する要素\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  件数範囲を表示する要素を削除\n";
    TXT += " *\n";
    TXT += " *  P.removeInfoEl(idx);\n";
    TXT += " *    @param {number} idx - 削除したいインデックス番号\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  現在の件数範囲を表示する要素を取得\n";
    TXT += " *\n";
    TXT += " *  P.getInfoEl();\n";
    TXT += " *    @return {HTMLElement[]} 件数範囲を表示する要素\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    console.debug(['pagination.js', 'ページネーションを管理・描画するクラス', TXT.split("\n")]);
  }

  /**
   * コンストラクタ
   *
   *  @param {HTMLElement[]} controlsSelector - ページ操作UIを表示する要素
   *  @param {?HTMLElement[]} infoSelector - 件数範囲を表示する要素
   *  @param {?number} totalItems - 総件数
   *  @param {?number} itemsPerPage - １ページに表示する件数
   *  @param {?number} currentPage - 現在のページ番号
   *  @param {?callback} onPageChangeCallback - ページ番号をクリックした時
   */
  constructor(controlsSelector, infoSelector, totalItems, itemsPerPage, currentPage, onPageChangeCallback) {
    /** @type {HTMLElement[]} ページ操作UIを表示する要素 */
    this.controlsEl = [];
    if (controlsSelector) this.addControlsEl(controlsSelector);

    /** @type {HTMLElement[]} 件数範囲を表示する要素 */
    this.infoEl = [];
    if (infoSelector) this.addInfoEl(infoSelector);

    /** @type {number} 総件数 */
    this.totalItems = totalItems ? Number(totalItems) : undefined;

    /** @type {number} １ページに表示する件数 */
    this.itemsPerPage = itemsPerPage ? Number(itemsPerPage) : undefined;

    /** @type {number} 現在のページ番号 */
    this.currentPage = undefined;

    /**
     * ページが切り替わったときに呼び出されるコールバック関数。
     *
     * @callback onPageChangeCallback
     * @param {number} page - 現在のページ番号
     */
    this.onPageChange = typeof onPageChangeCallback === "function" ? onPageChangeCallback : null;


    if (currentPage) this.goToPage(currentPage);
  }


  /**
   * 指定したページ番号へ移動する
   *
   *  @param {number} page - 遷移先のページ番号
   */
  goToPage(page) {
    const totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
    if (Number(page) >= 1 && Number(page) <= totalPages) {
      this.currentPage = Number(page);
      this.update();
    }
  }


  /**
   * データをセット
   *
   *  @param {number} totalItems - 総件数
   *  @param {number} itemsPerPage - １ページに表示する件数
   *  @param {number} currentPage - 現在のページ番号
   */
  reset(totalItems, itemsPerPage, currentPage) {
    /** @type {number} 総件数 */
    this.totalItems = totalItems ? Number(totalItems) : undefined;

    /** @type {number} １ページに表示する件数 */
    this.itemsPerPage = itemsPerPage ? Number(itemsPerPage) : undefined;

    /** @type {number} 現在のページ番号 */
    this.goToPage(currentPage ? currentPage : 1);
  }

  /**
   * 再描画
   */
  update() {
    this.#renderControls();
    this.#renderInfo();
  }

  /**
   * ページ操作UIを表示する要素を追加
   *
   *  @param {HTMLElement[]} El - ページ操作UIを表示する要素
   */
  addControlsEl(El) {
    if (El instanceof HTMLElement) {
      this.controlsEl.push(El);
    } else if (
      El instanceof NodeList ||
      El instanceof HTMLCollection ||
      (Array.isArray(El) && El.every(el => el instanceof HTMLElement))
    ) {
      for (const el of El) {
        this.controlsEl.push(el);
      };
    }
    this.#renderControls();
  }

  /**
   * ページ操作UIを表示する要素を削除
   *
   *  @param {number} idx - 削除したいインデックス番号
   */
  removeControlsEl(idx) {
    this.controlsEl.splice(idx, 1);
    this.#renderControls();
  }

  /**
   * ページ操作UIを表示する要素を取得
   *
   *  @return {HTMLElement[]} ページ操作UIを表示する要素
   */
  getControlsEl() {
    return this.controlsEl;
  }

  /**
   * ページ操作UIを表示する
   *
   *  @private
   */
  #renderControls(pageGroup) {
    if (!this.currentPage) return;
    const controlsHtml = this.#renderControlsHtml(pageGroup);
    const controlsCss = this.#renderControlsCSS();

    for (const [idx, el] of this.controlsEl.entries()) {
      el.innerHTML = controlsHtml;
      if (idx === 0) el.appendChild(controlsCss);
      el.querySelectorAll("li").forEach(liEl => {
        liEl.addEventListener("click", (e) => this.#ControlButtonClick(e));
      });
    }
  }
  #renderControlsHtml(pageGroup) {
    const totalPages = Math.ceil(this.totalItems / this.itemsPerPage);
    const pageGroupCount = pageGroup ? Number(pageGroup) : Math.ceil(this.currentPage / 10);
    let html = "<ul class=\"YsPagination\">";

    // 「前へ」ボタン
    if (pageGroupCount > 1) html += `<li class="cursor" data-group="${pageGroupCount - 1}">‹</li>`;

    // ページ番号ボタン
    let startItem = Math.max(1, (pageGroupCount - 1) * 10 + 1);
    let endItem = Math.min(totalPages, startItem + 9);
    for (let i=startItem; i<=endItem; i++) {
      html += `<li class="btn ${i === this.currentPage ? "disabled" : ""}" data-page="${i}">${i}</li>`;
    }

    // 「次へ」ボタン
    if (pageGroupCount * 10 < totalPages) html += `<li class="cursor" data-group="${pageGroupCount + 1}">›</li>`;

    html += "</ul>";
    return html;
  }
  #renderControlsCSS() {
    const styleEl = document.createElement("style");
    styleEl.textContent = `
      ul.YsPagination {
        display: inline-block;
        margin: 0;
        padding: 0;
        list-style-type: none;
        user-select: none;
      }
      ul.YsPagination > li {
        display: inline-block;
        border-style: solid;
        border-color: #666;
        background-color: #fff;
        text-align: center;
        vertical-align: middle;
        cursor: pointer;
      }
      ul.YsPagination > li.disabled {
        border: none;
        font-weight: bold;
        cursor: default;
      }
      @media print, screen and (min-width: 1025px) {
        ul.YsPagination > li {
          width: 35px;
          height: 35px;
          margin: 0 5px;
          border-width: 1px;
          border-radius: 4px;
          font-size: 14px;
          line-height: 35px;
        }
        ul.YsPagination > li.cursor {
          font-size: 24px;
        }
      }
      @media screen and (min-width: 641px) and (max-width: 1024px) {
        ul.YsPagination > li {
          width: 4.0983vw;
          height: 4.0983vw;
          margin: 0 0.5854vw;
          border-width: 0.117vw;
          border-radius: 0.4683vw;
          font-size: 1.6393vw;
          line-height: 4.0983vw;
        }
        ul.YsPagination > li.cursor {
          font-size: 2.8103vw;
        }
      }
      @media screen and (max-width: 640px) {
        ul.YsPagination {
          display: none;
        }
      }
    `;
    return styleEl;
  }
  #ControlButtonClick(e) {
    const dataSet = e.target.dataset;
    if(dataSet?.group) {
      this.#renderControls(dataSet.group);
      return;
    }
    if (dataSet?.page) {
      if (typeof this.onPageChange === "function") this.onPageChange(dataSet.page);
      return;
    }
  }


  /**
   * 件数範囲を表示する要素を追加
   *
   *  @param {HTMLElement[]} El - 件数範囲を表示する要素
   */
  addInfoEl(El) {
    if (El instanceof HTMLElement) {
      this.infoEl.push(El);
    } else if (
      El instanceof NodeList ||
      El instanceof HTMLCollection ||
      (Array.isArray(El) && El.every(el => el instanceof HTMLElement))
    ) {
      for (const el of El) {
        this.infoEl.push(el);
      };
    }
    this.#renderInfo();
  }

  /**
   * 件数範囲を表示する要素を削除
   *
   *  @param {number} idx - 削除したいインデックス番号
   */
  removeInfoEl(idx) {
    this.infoEl.splice(idx, 1);
    this.#renderInfo();
  }

  /**
   * 現在の件数範囲を表示する要素を取得
   *
   *  @return {HTMLElement[]} 件数範囲を表示する要素
   */
  getInfoEl() {
    return this.infoEl;
  }

  /**
   * 現在のページに応じた件数範囲を表示する
   *
   *  @private
   */
  #renderInfo() {
    if (!this.currentPage) return;
    const start = (this.currentPage - 1) * this.itemsPerPage + 1;
    const end = Math.min(this.totalItems, this.currentPage * this.itemsPerPage);
    const text = (start ? Number(start).toLocaleString() : "--") + " 件 ～ " + (end ? Number(end).toLocaleString() : "--") + " 件目を表示（全 " + (this.totalItems ? Number(this.totalItems).toLocaleString() : "--") + " 件）";
    for (const el of this.infoEl) {
      el.textContent = text;
    }
  }
}
new Pagination().help();