;
/*************************************************************
 *
 * Copyright (c) 2025 ysrock Co., Ltd.	<info@ysrock.co.jp>
 * Copyright (c) 2025 Yasuo Sugano	<sugano@ysrock.co.jp>
 *
 * Version	: 1.0.1
 * Update	  : 2025.10.07
 *
 ************************************************************/
'use strict';

class MultiTask{
  /**
   * コンストラクタ
   *
   *  @param {boolean|callback} abortHandler? - true|callbackで×ボタンを表示（中止処理）
   */
  constructor(abortHandler) {
    this.abortHandler = abortHandler;
    this.id = null;
    this.isAbort = null;
  }

  help() {
    let TXT = " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  Copyright (c) 2025 ysrock Co., Ltd. <info@ysrock.co.jp>\n";
    TXT += " *  Copyright (c) 2025 Yasuo Sugano <sugano@ysrock.co.jp>\n";
    TXT += " *\n";
    TXT += " *  Version : 1.0.0\n";
    TXT += " *  update  : 2025.10.06\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  マルチタスク\n";
    TXT += " *    const MT = new MultiTask(abortHandler)\n";
    TXT += " *      @param {boolean|callback} abortHandler? - true|callbackで×ボタンを表示（中止処理）\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  マルチタスクを実行する\n";
    TXT += " *    MT.execute(...args)\n";
    TXT += " *      @param {function} args[].func - 実行する関数\n";
    TXT += " *      @param {array|undefined} args[].args? - 関数に渡す引数\n";
    TXT += " *      @param {string} args[].html? - 関数実行中の文言\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  マルチタスクを中止する\n";
    TXT += " *    MT.abort()\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    console.debug(['multiTask.js', 'マルチタスクを実行するクラス', TXT.split("\n")]);
  }

  /**
   * マルチタスクを実行する
   *
   *  @param {object[]} tasks - マルチタスク
   *  @param {function} tasks[].func - 実行する関数
   *  @param {array} tasks[].args? - 関数に渡す引数
   *  @param {string} tasks[].html? - 関数実行中の文言
   */
  execute(tasks) {
    return new Promise((resolve, reject) => {
      const elem = this.#createElement(tasks);
      document.body.appendChild(elem);
      document.body.classList.add("noScrollY_" + this.id);
      if (typeof(this.abortHandler) == "function" || this.abortHandler === true) document.getElementById(this.id).querySelector("div.close").addEventListener("click", (...args) => this.abort());

      Promise.all(tasks.map((seriesTaskAry, seriesTaskIdx) => this.seriesTaskExecute(seriesTaskAry, seriesTaskIdx) ))
      .then((result) => {
        console.debug('Promise.all', 'result', result);
        resolve(result);
        setTimeout(() => this.destroy(), 1250);
      }).catch((e) => reject(e));
    });
  }

  /**
   * 要素の作成
   *
   *  @param {function} tasks[].func - 実行する関数
   *  @param {array} tasks[].args? - 関数に渡す引数
   *  @param {string} tasks[].html? - 関数実行中の文言
   */
  #createElement(tasks) {
    this.id = this.#makeUUID();
    const elem = document.createElement("div");
    elem.classList.add("multiTaskWrap");
    elem.innerHTML = this.#makeHtml(tasks);
    elem.id = this.id;
    elem.appendChild(this.#makeCSS());
    return elem;
  }

  /**
   * UUIDを作成
   *  @return {string}
   */
  #makeUUID() {
    return "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(/[xy]/g, function(c){
      let r = Math.random() * 16 | 0, v = c == "x" ? r : r & 3 | 8;
      return v.toString(16);
    });
  }

  /**
   * HTMLを作成
   *
   *  @param {function} tasks[].func - 実行する関数
   *  @param {array} tasks[].args? - 関数に渡す引数
   *  @param {string} tasks[].html? - 関数実行中の文言
   */
  #makeHtml(tasks) {
    let html = "";
    html += "<div class=\"popupWrap\">";
    for (let i=0, len=tasks.length; i<len; i++) {
      html += " <div class=\"tasks\">";
      html += "  <div class=\"bar\"><div></div></div>";
      html += "  <div class=\"html\"></div>";
      html += " </div>";
    };
    html += "</div><!-- END div.popupWrap -->";
    if (typeof(this?.abortHandler) == "function" || this?.abortHandler === true) html += "<div class=\"close\"></div>";
    return html;
  }

  /**
   * スタイルシートを作成
   *
   *  @return {element} - style
   */
  #makeCSS() {
    const style = document.createElement("style");
    style.textContent = `
      div.multiTaskWrap {
        position: fixed;
        top: 0;
        left: 0;
        z-index: 10000;
        width: 100vw;
        height: 100vh;
        background-color: rgba(0, 0, 0, .6);
      }
      div.multiTaskWrap > div.popupWrap {
        position: fixed;
        top: 10vh;
        left: 50%;
        transform: translateX(-50%);
        background-color: #fff;
        box-shadow: .1em .1em .5em .2em;
        padding: 3em 2em;
        border-radius: .4em;
        text-align: center;
      }
      div.multiTaskWrap > div.popupWrap > div.tasks {
        text-align: left;
      }
      div.multiTaskWrap > div.popupWrap > div.tasks + div.tasks {
        margin-top: 1em;
      }
      div.multiTaskWrap > div.popupWrap > div.tasks > div.bar {
        height: 1.2em;
        border: .1em solid #ddd;
        background-color: #f8f8f8;
      }
      div.multiTaskWrap > div.popupWrap > div.tasks > div.bar > div {
        width: 0%;
        height: 100%;
        background-color: #9d9de9;
        transition: width 1s ease-in-out;
      }
      div.multiTaskWrap > div.popupWrap > div.tasks > div.html {
        padding-top: .5em;
        text-align: center;
      }

      div.multiTaskWrap > div.close {
        position: fixed;
        top: 1em;
        right: 1em;
        display: inline-block;
        width: 2em;
        height: 2em;
        cursor: pointer;
      }
      div.multiTaskWrap > div.close::before,
      div.multiTaskWrap > div.close::after {
        content: "";
        display: inline-block;
        width: 2em;
        height: .4em;
        background-color: #fff;
        position: absolute;
        top: 50%;
        left: 50%;
      }
      div.multiTaskWrap > div.close::before {
        transform: translate(-50%, -50%) rotate(45deg);
      }
      div.multiTaskWrap > div.close::after {
        transform: translate(-50%, -50%) rotate(-45deg);
      }

      @media print, screen and (min-width: 1025px) {
        div.multiTaskWrap > div.popupWrap {
          width: 60vw;
        }
      }
      @media screen and (min-width: 641px) and (max-width: 1024px) {
        div.multiTaskWrap > div.popupWrap {
          width: 75vw;
        }
      }
      @media screen and (max-width: 640px) {
        div.multiTaskWrap > div.popupWrap {
          width: 80vw;
        }
      }

      body[class*="noScrollY"] {
        overflow-y: hidden;
      }
    `;
    return style;
  }

  /**
   * 直列処理を順番に実行
   *
   *  @param {object[]} seriesTaskAry - 直列に処理するタスク
   *  @param {string} seriesTaskAry[].func - 実行する関数
   *  @param {string} seriesTaskAry[].args? - 関数に渡す引数
   *  @param {string} seriesTaskAry[].html? - 関数実行中の文言
   *  @param {int} seriesTaskIdx - 並列処理のインデックス
   */
  seriesTaskExecute(seriesTaskAry, seriesTaskIdx) {
    return new Promise(async (resolve, reject) => {
      const resultAry = [];
      for (let i=0, len=seriesTaskAry.length; i<len; i++) {
        if (this.isAbort) {
          if (typeof(this.abortHandler) == "function") this.abortHandler(seriesTaskAry[i]);
          resultAry.push('aborted.');
          return resolve(resultAry);
        }

        const taskObj = seriesTaskAry[i];
        const nowTask = i;
        const maxTask = seriesTaskAry.length;
        this.#changeProgressMessage(seriesTaskIdx, taskObj?.html !== undefined ? taskObj?.html : "");
        this.#changeProgressBar(seriesTaskIdx, nowTask / maxTask * 100);
        resultAry.push(await taskObj?.func(taskObj?.args));
      }

      this.#changeProgressBar(seriesTaskIdx, 100);
      this.#changeProgressMessage(seriesTaskIdx, "処理が完了しました。");
      resolve(resultAry);
    });
  }

  /**
   * 進捗バーを変更
   *
   *  @param {int} taskIdx - タスクインデックス
   *  @param {float} per - パーセント
   */
  #changeProgressBar(taskIdx, per) {
    if (this.isAbort) return;
    const barElem = document.getElementById(this.id).querySelector("div.popupWrap > div.tasks:nth-child(" + (Number(taskIdx) + 1) + ") > div.bar > div");
    if (barElem === null) return;
    barElem.style.width = per + "%";
  }

  /**
   * 進捗メッセージを変更
   */
  #changeProgressMessage(taskIdx, html) {
    if (this.isAbort) return;
    const messageElem = document.getElementById(this.id).querySelector("div.popupWrap > div.tasks:nth-child(" + (Number(taskIdx) + 1) + ") > div.html");
    if (messageElem === null) return;
    messageElem.innerHTML = html;
  }

  /**
   * マルチタスクを非表示
   */
  destroy() {
    const elem = document.getElementById(this.id);
    if (elem) document.getElementById(this.id).remove();
    document.body.classList.remove("noScrollY_" + this.id);
  }

  /**
   * 中止する
   */
  abort() {
    this.isAbort = true;
    this.destroy();
  }

}
new MultiTask().help();
