;
/*************************************************************
 *
 * Copyright (c) 2025 ysrock Co., Ltd.	<info@ysrock.co.jp>
 * Copyright (c) 2025 Yasuo Sugano	<sugano@ysrock.co.jp>
 *
 * Version	: 1.0.2
 * Update	  : 2025.09.30
 *
 ************************************************************/
'use strict';


class Dialog {
  constructor () {
    this.id = null;
  }

  help() {
    let TXT = " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  Copyright (c) 2025 ysrock Co., Ltd. <info@ysrock.co.jp>\n";
    TXT += " *  Copyright (c) 2025 Yasuo Sugano <sugano@ysrock.co.jp>\n";
    TXT += " *\n";
    TXT += " *  Version : 1.0.2\n";
    TXT += " *  update  : 2025.09.30\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ダイアログを表示\n";
    TXT += " *    new Dialog().open(inObj)\n";
    TXT += " *      @param {object} inObj.header - ヘッダーに表示させるもの\n";
    TXT += " *      @param {string} inObj.header.subject - 件名\n";
    TXT += " *      @param {object} inObj.body - メイン部分に表示させるもの\n";
    TXT += " *      @param {string} inObj.body.html - 本文\n";
    TXT += " *      @param {object} inObj.footer - フッター部分に表示させるもの\n";
    TXT += " *      @param {object[]} inObj.footer.buttons - ボタン配列\n";
    TXT += " *      @param {string} inObj.footer.buttons[].text - 表示文言\n";
    TXT += " *      @param {callback} inObj.footer.buttons[].click - クリック時のコールバック\n";
    TXT += " *      @param {string[]} inObj.footer.buttons[].class - 追加するクラス名\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    TXT += " *\n";
    TXT += " *  ダイアログを閉じる\n";
    TXT += " *    new Dialog().close()\n";
    TXT += " *\n";
    TXT += " ************************************************************\n";
    console.debug(['dialog.js', 'ダイアログを表示するクラス', TXT.split("\n")]);
  }

  /**
   * ダイアログを表示
   *  @param {object{}} inObj
   *  @param {object} inObj.header - ヘッダーに表示させるもの
   *  @param {string} inObj.header.subject - 件名
   *  @param {object} inObj.body - メイン部分に表示させるもの
   *  @param {string} inObj.body.html - 本文
   *  @param {object} inObj.footer - フッター部分に表示させるもの
   *  @param {object[]} inObj.footer.buttons - ボタン配列
   *  @param {string} inObj.footer.buttons[].text - 表示文言
   *  @param {callback} inObj.footer.buttons[].click - クリック時のコールバック
   *  @param {string[]} inObj.footer.buttons[].class - 追加するクラス名
   *  @param {boolean} inObj.footer.buttons[].disabled - 非活性化
   */
  open(inObj) {
    const elem = this.#createElement(inObj);
    document.body.appendChild(elem);
    document.getElementById(this.id).querySelector("div.popupWrap > div.header > div.close").addEventListener("click", (...args) => this.close(...args));
    this.#footerButtonEvent(inObj?.footer?.buttons);
    document.body.classList.add("noScrollY_" + this.id);
  }

  /**
   * 要素の作成
   *  @param {object{}} inObj
   *  @param {object} inObj.header - ヘッダーに表示させるもの
   *  @param {string} inObj.header.subject - 件名
   *  @param {object} inObj.body - メイン部分に表示させるもの
   *  @param {string} inObj.body.html - 本文
   *  @param {object} inObj.footer - フッター部分に表示させるもの
   *  @param {object[]} inObj.footer.buttons - ボタン配列
   *  @param {string} inObj.footer.buttons[].text - 表示文言
   *  @param {callback} inObj.footer.buttons[].click - クリック時のコールバック
   *  @param {string[]} inObj.footer.buttons[].class - 追加するクラス名
   *  @param {boolean} inObj.footer.buttons[].disabled - 非活性化
   */
  #createElement(inObj) {
    this.id = this.#makeUUID();
    const elem = document.createElement("div");
    elem.classList.add("dialogWrap");
    elem.innerHTML = this.#makeHtml(inObj);
    elem.id = this.id;
    elem.appendChild(this.#makeCSS());
    return elem;
  }

  /**
   * UUIDを作成
   *  @return {string}
   */
  #makeUUID() {
    return "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(/[xy]/g, function(c){
      let r = Math.random() * 16 | 0, v = c == "x" ? r : r & 3 | 8;
      return v.toString(16);
    });
  }

  /**
   * HTMLを作成
   *  @param {object{}} inObj
   *  @param {object} inObj.header - ヘッダーに表示させるもの
   *  @param {string} inObj.header.subject - 件名
   *  @param {object} inObj.body - メイン部分に表示させるもの
   *  @param {string} inObj.body.html - 本文
   *  @param {object} inObj.footer - フッター部分に表示させるもの
   *  @param {object[]} inObj.footer.buttons - ボタン配列
   *  @param {string} inObj.footer.buttons[].text - 表示文言
   *  @param {callback} inObj.footer.buttons[].click - クリック時のコールバック
   *  @param {string[]} inObj.footer.buttons[].class - 追加するクラス名
   *  @param {boolean} inObj.footer.buttons[].disabled - 非活性化
   */
  #makeHtml(inObj) {
    let html = "";
    html += "<div class=\"popupWrap\">";
    html += " <div class=\"header\">";
    html += "  <div class=\"subject\">" + (inObj?.header?.subject !== undefined ? inObj.header.subject : "") + "</div>";
    html += "  <div class=\"close\"></div>";
    html += " </div><!-- END div.header -->";
    html += " <div class=\"body\">";
    html += "  <div class=\"html\">" + (inObj?.body?.html !== undefined ? inObj.body.html : "") + "</div>";
    html += " </div><!-- END div.body -->";
    html += " <div class=\"footer\">";
    if (inObj?.footer?.buttons !== undefined && Array.isArray(inObj.footer.buttons)) {
      for (let i=0, len=inObj.footer.buttons.length; i<len; i++) {
        const attrClass = inObj.footer.buttons[i].class !== undefined && Array.isArray(inObj.footer.buttons[i].class) ? " class=\"" + inObj.footer.buttons[i].class.join(" ") + "\"" : "";
        const attrDisabled = inObj.footer.buttons[i].disabled !== undefined && inObj.footer.buttons[i].disabled ? " disabled=\"" + inObj.footer.buttons[i].disabled + "\"" : "";
        html += "  <button type=\"button\"" + attrClass + attrDisabled + ">" + (inObj.footer.buttons[i].text !== undefined ? inObj.footer.buttons[i].text : "") + "</button>";
      }
    }
    html += " </div><!-- END div.footer -->";
    html += "</div><!-- END div.popupWrap -->";
    return html;
  }

  /**
   * スタイルシートを作成
   *  @return {element} - style
   */
  #makeCSS() {
    const style = document.createElement("style");
    style.textContent = `
      div.dialogWrap {
        position: fixed;
        top: 0;
        left: 0;
        z-index: 10000;
        width: 100vw;
        height: 100vh;
        background-color: rgba(0, 0, 0, .2);
        text-align: left;
      }
      div.dialogWrap > div.popupWrap {
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        box-shadow: 0 8px 40px rgba(22, 37, 51, .2);
        min-width: 30vw;
        max-width: 95vw;
      }
      div.dialogWrap > div.popupWrap > div.header {
        display: flex;
        flex-direction: row;
        background-color: #fff;
        border-bottom: 1px solid #d9e0e8;
        border-top-left-radius: 8px;
        border-top-right-radius: 8px;
        padding: 12px 16px;
        font-size: 18px;
        line-height: 1em;
        font-weight: bold;
        color: #000;
      }
      div.dialogWrap > div.popupWrap > div.header > div.subject {
        flex: 1;
        padding: 8px 0;
      }
      div.dialogWrap > div.popupWrap > div.header > div.close {
        position: relative;
        width: 20px;
        height: 20px;
        margin: auto;
        cursor: pointer;
      }
      div.dialogWrap > div.popupWrap > div.header > div.close::before,
      div.dialogWrap > div.popupWrap > div.header > div.close::after {
        content: "";
        position: absolute;
        top: 0;
        left: 0;
        bottom: 0;
        right: 0;
        width: 18px;
        height: 3px;
        margin: auto;
        background-color: #858c90;
      }
      div.dialogWrap > div.popupWrap > div.header > div.close::before {
        transform: rotate(45deg);
      }
      div.dialogWrap > div.popupWrap > div.header > div.close::after {
        transform: rotate(-45deg);
      }
      div.dialogWrap > div.popupWrap > div.body {
        min-height: 150px;
        max-height: 300px;
        background-color: #fff;
        padding: 24px 24px 48px;
        font-size: 18px;
        word-break: break-all;
        overflow-x: hidden;
        overflow-y: auto;
      }
      div.dialogWrap > div.popupWrap > div.body > div.html > label {
        cursor: pointer;
      }
      div.dialogWrap > div.popupWrap > div.body > div.html > label > input[type=checkbox] {
        margin-right: .5em;
        vertical-align: inherit;
      }
      div.dialogWrap > div.popupWrap > div.footer {
        display: flex;
        justify-content: flex-end;
        border-top: 1px solid #d9e0e8;
        border-bottom-left-radius: 8px;
        border-bottom-right-radius: 8px;
        background-color: #f9fbfd;
        padding: 12px 16px;
      }
      div.dialogWrap > div.popupWrap > div.footer > button {
        height: 40px;
        margin-left: 8px;
        padding: 0 16px;
        background-color: #f8f8f8;
        border-radius: 4px;
        border: 1px solid #d9e0e8;;
        font-size: 16px;
        font-weight: bold;
        color: #555d65;
        cursor: pointer;
        outline: none;
        word-break: keep-all;
        transition: background-color .2s, color .2s;
      }
      div.dialogWrap > div.popupWrap > div.footer > button:disabled {
        cursor: default;
        opacity: .6;
      }
      div.dialogWrap > div.popupWrap > div.footer > button:not(:disabled):hover {
        background-color: #f0f0f0;
      }
      div.dialogWrap > div.popupWrap > div.footer > button.blue {
        background-color: #005bac;
        color: #fff;
      }
      div.dialogWrap > div.popupWrap > div.footer > button:not(:disabled).blue:hover {
        background-color: #166fbd;
        color: #ddd;
      }
      div.dialogWrap > div.popupWrap > div.footer > button.red {
        background-color: #cf0084;
        color: #fff;
      }
      div.dialogWrap > div.popupWrap > div.footer > button:not(:disabled).red:hover {
        background-color: #e941abff;
        color: #ddd;
      }

      body[class*="noScrollY"] {
        overflow-y: hidden;
      }
    `;
    return style;
  }

  /**
   * ヘッダーの閉じるイベント
   */

  /**
   * フッターのボタンイベント
   *  @param {object[]} buttons - ボタン配列
   *  @param {callback} buttons[].click - クリック時のコールバック
   */
  #footerButtonEvent(buttons) {
    if (buttons === undefined || !Array.isArray(buttons)) return;
    for (let i=0, len=buttons.length; i<len; i++) {
      if (buttons[i].click === undefined) continue;
      const elem = document.getElementById(this.id).querySelector("div.popupWrap > div.footer > button:nth-of-type(" + (i+1) + ")");
      elem.addEventListener("click", (...args) => buttons[i].click(...args));
    }
  }

  /**
   * ダイアログを閉じる
   */
  close () {
    const elem = document.getElementById(this.id);
    if (elem) document.getElementById(this.id).remove();
    document.body.classList.remove("noScrollY_" + this.id);
  }

  /**
   * 本文要素
   */
  getBodyElement() {
    return document.getElementById(this.id).querySelector("div.popupWrap > div.body > div.html");
  }

  /**
   * ボタン要素
   */
  getButtonElements() {
    return document.getElementById(this.id).querySelectorAll("div.popupWrap > div.footer > button");
  }
}
new Dialog().help();
