;
/*************************************************************
 *
 * Copyright (c) 2025 ysrock Co., Ltd.	<info@ysrock.co.jp>
 * Copyright (c) 2025 Yasuo Sugano	<sugano@ysrock.co.jp>
 *
 * Version	: 1.0.0
 * Update	  : 2025.02.19
 *
 *  peerClosure.ready( terminal, iceServers );
 *
 ************************************************************/
'use strict';
  
const peerClosure = (() => {
  return {
    terminal: null,
    dir: null,
    peerConnection: null,
    peerConnectionConfig: null,


    /**
     * 呼び出し方
     *  @param terminal (enum) parent|child
     *  @param ?peerConnectionConfig  = null (object)
     *  @param ?setupDataChannel = this.setupDataChannel (callback)
     *  @param ?ConnectionStateChange = this.ConnectionStateChange (callback)
     */
    ready: function (terminal, peerConnectionConfig, setupDataChannel, ConnectionStateChange) {
      this.terminal = terminal && terminal == "child" ? terminal : "parent";
      this.dir = this.getDIR();
      this.peerConnectionConfig = peerConnectionConfig ? peerConnectionConfig : null;
      if (setupDataChannel) this.setupDataChannel = setupDataChannel;
      if (ConnectionStateChange) this.ConnectionStateChange = ConnectionStateChange;

      if (this.terminal == "parent") this.makeSDP();
      else this.getSDP();
    },


    /**
     * メッセージを送信する
     *  @param message (string)
     */
    send: function (message) {
      if (!this.peerConnection || this.peerConnection.connectionState != 'connected') {
        console.error('PeerConnection is not established.');
        return false;
      };
      this.dataChannel.send(message);
    },


    /**
     * SDPを保存する
     */
    SaveSDP: function (SDP) {
      return new Promise((resolve, reject) => {
        fetch(this.dir + "save.php", {
          method: 'POST',
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify(SDP)
        }).then(res => {
          if (!res.ok) throw new Error("Network response was not ok.");
          return res.json();
        }).then((json) => {
          resolve(json);
        }).catch(err => {
          console.error([ 'commons/p2p.js', 'fetch', err ]);
          reject(err);
        });
      });
    },


    /**
     * 親端末：SDPを発行する
     */
    makeSDP: function () {
      // 新しい RTCPeerConnection を作成する
      this.peerConnection = this.createPeerConnection({});
      // Data channel を生成
      this.dataChannel = this.peerConnection.createDataChannel("dataChannel", { ordered: false });
      peerClosure.setupDataChannel(this.dataChannel);
      // Offer を生成する
      this.peerConnection.createOffer().then((sessionDescription) => {
        return this.peerConnection.setLocalDescription(sessionDescription);
      }).then(function () {
      }).catch(function (err) {
        console.error("setLocalDescription() failed.", err);
      });
    },


    /**
     * SDPを取得する
     */
    getSDP: function () {
      setTimeout(async() => {
        const resultObj = await this.readSDP();
        if (this.terminal != "parent" && resultObj !== null && 'parent' in resultObj !== false && resultObj.parent) this.setSDP(resultObj.parent);
        else if (this.terminal == "parent" && resultObj !== null && 'child' in resultObj !== false && resultObj.child) this.setSDP(resultObj.child);
        else this.getSDP();
      }, 1000);
    },
    readSDP: function () {
      return new Promise((resolve, reject) => {
        fetch(this.dir + "read.php", {
          method: 'GET',
          headers: { "Content-Type": "application/json" },
        }).then(res => {
          if (!res.ok) throw new Error("Network response was not ok.");
          return res.json();
        }).then((json) => {
          resolve(json === null ? null : JSON.parse(json));
        }).catch(err => {
          console.error([ 'commons/p2p.js', 'fetch', err ]);
          reject(err);
        });
      });
    },


    /**
     * 取得したSDPをセットする
     */
    setSDP: function (SDP) {
      if (this.terminal == "parent") {
        const answer = new RTCSessionDescription({
          type: 'answer',
          sdp: SDP,
        });
        this.peerConnection.setRemoteDescription(answer).then(function() {
          console.debug('setRemoteDescription() succeeded.');
        }).catch(function(err) {
          console.error('setRemoteDescription() failed.', err);
        });
        return;
      } else {
        const  offer = new RTCSessionDescription({
          type: 'offer',
          sdp: SDP,
        });
        // 新しい RTCPeerConnection を作成する
        this.peerConnection = this.createPeerConnection({'parent': SDP });
        this.peerConnection.setRemoteDescription(offer).then(function() {
          console.debug('setRemoteDescription() succeeded.');
        }).catch(function(err) {
          console.error('setRemoteDescription() failed.', err);
        });
        this.peerConnection.createAnswer().then((sessionDescription) => {
          console.debug('createAnswer() succeeded.');
          return this.peerConnection.setLocalDescription(sessionDescription);
        }).then(function() {
          console.debug('setLocalDescription() succeeded.');
        }).catch(function(err) {
          console.error('setLocalDescription() failed.', err);
        });
        return;
      };
    },


    createPeerConnection: function (sdpObj) {
      const pc = new RTCPeerConnection(this.peerConnectionConfig);
      pc.onicecandidate = this.ICEcandidate(pc, sdpObj);
      pc.onconnectionstatechange = this.ConnectionStateChange(pc);
      pc.ondatachannel  = this.DataChannel();
      return pc;
    },
    ICEcandidate: function (pc, sdpObj) {
      return async (evt) => {
        if (evt.candidate) return console.debug([ 'commons/p2p.js', 'peerClosure.ICEcandidate', 'Collecting ICE candidates', evt.candidate ]);
        if (this.terminal == "parent") {
          sdpObj = { parent: pc.localDescription.sdp };
          console.debug([ 'commons/p2p.js', 'sdpObj', sdpObj ]);
          await  this.SaveSDP(sdpObj);
          this.getSDP();
          return;
        }
        else {
          sdpObj['child'] = pc.localDescription.sdp;
          console.debug([ 'commons/p2p.js', 'sdpObj', sdpObj ]);
          await  this.SaveSDP(sdpObj);
          return;
        };
      };
    },
    ConnectionStateChange: (pc) => {
      return function (evt) {
        console.debug([ 'commons/p2p.js', 'peerClosure.ConnectionStateChange', evt, pc ]);
      };
    },
    DataChannel: () => {
      return function (evt) {
        peerClosure.setupDataChannel(evt.channel);
        peerClosure.dataChannel = evt.channel;
      };
    },
    setupDataChannel: function (dc) {
      dc.onerror = (e) => console.debug([ 'commons/p2p.js', 'peerClosure.setupDataChannel onerror', e ]);
      dc.onmessage = (e) => {
        console.debug([ 'commons/p2p.js', 'peerClosure.setupDataChannel onmessage', e ]);
        // vueClosure.vueApp.recvMessage = "> " + e.data + "\n" + vueClosure.vueApp.recvMessage;
      };
      dc.onopen = (e) => console.debug([ 'commons/p2p.js', 'peerClosure.setupDataChannel onopen', e ]);
      dc.onclose = (e) => {
        console.debug([ 'commons/p2p.js', 'peerClosure.setupDataChannel onclose', e ]);
        location.reload();
      };
    },


    /**
     * このファイルがあるディレクトリを取得
     */
    getDIR: function() {
      const scripts = document.getElementsByTagName("script");
      let i = scripts.length;
      while (i--) {
        let match = scripts[i].src.match(/(^|.*\/)p2p\.js$/);
        if (match[1]) return match[1];
      };
    },

  }
})();
